<?php
/*

Copyright (c)2003 DuckCorp(tm) and RtpNet(tm)



This file is part of DFTK 

DFTK is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

DFTK is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with DFTK  if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/


/**
 * @package		dftk
 * @author		Duck <duck@DuckCorp.org>
 * @author		Rtp <rtp@rtp-net.org>
 * @copyright	Copyright (c)2003 DuckCorp(tm) and RtpNet(tm)
 * @license		http://www.gnu.org/licenses/gpl.html					GNU General Public License
 * @version		0.4.0
 */


/**
 * DFTK IDA* algorithm
 *
 * @package			dftk
 * @subpackage		dftk-ai
 * @author			Duck <duck@DuckCorp.org>
 *
 * @access			public
 */

class DftkAiIDAstar
{
	/**
	 * Trace Manager
	 *
	 * @access	private
	 * @var		string
	 */
	var	$_tracemgr;

	/**
	 * Node list
	 *
	 * @access	private
	 * @var		array
	 */
	var	$_node_list;

	/**
	 * Cost function name
	 *
	 * @access	private
	 * @var		string
	 */
	var	$_cost_fct;

	/**
	 * Heuristic function name
	 *
	 * @access	private
	 * @var		string
	 */
	var	$_heuristic_fct;

	/**
	 * Successor selection function name
	 *
	 * @access	private
	 * @var		string
	 */
	var	$_successor_selection_fct;

	/**
	 * Start node
	 *
	 * @access	private
	 * @var		string
	 */
	var	$_start_node;

	/**
	 * Stop node
	 *
	 * @access	private
	 * @var		string
	 */
	var	$_stop_node;

	/**
	 * Routes cache
	 *
	 * @access	private
	 * @var		array
	 */
	var	$_routes_cache;

	/**
	 * Constructor
	 *
	 * @access		public
	 * @param		object DftkDaTraceManager	&$tracemgr			Language Manager
	 */
	function DftkAiIDAstar(&$tracemgr)
	{
		$this->_tracemgr =& $tracemgr;

		if (!$this->_tracemgr->is_module("DFTK-LDAP"))
			$this->_tracemgr->register_traces("DFTK-IA",DFTK_ROOT_PATH."/ai/traces/");
		
		register_shutdown_function(array(&$this, "_DftkAiIDAstar"));

		$this->_node_list = array();
	}

	/**
	 * Destructor
	 *
	 * @access		private
	 */
	function _DftkAiIDAstar()
	{
	}

	/**
	 * Set node list
	 *
	 * The node list contains the one-way roads to other nodes;
	 * to use both-way roads you have to specify two one-way roads.
	 *
	 * Node list is indexed by <node_name> and contains these fields :
	 *  + 'successors' : array
	 *  + 'data' : optional user field
	 *
	 * Each element of the 'successors' array is a couple (array) :
	 *  + <next_node_name>
	 *  + <road_name>
	 *
	 * The road name is only informational.
	 *
	 * The 'data' field can contain whatever data the user need to compute
	 * cost and heuritic values (eg. geographic position).
	 *
	 * @access		public
	 * @param		array					$node_list				Node list
	 * @return		object DftkDaTrace		$r						Trace
	 */
	function &set_node_list($node_list)
	{
		$r =& $this->_tracemgr->create_trace();

		$this->_node_list = $node_list;
		$this->_routes_cache = array();

		return $r;
	}

	/*
	 * Set the cost function name
	 *
	 * This function must accept the following parameters :
	 *  + node_list
	 *  + node
	 *	+ next_node
	 *
	 * It must return the cost to go from node to next_node.
	 *
	 * @access		public
	 * @param		array					$node_list				Node list
	 * @return		object DftkDaTrace		$r						Trace
	 */
	function &set_cost_fct($cost_fct)
	{
		$r =& $this->_tracemgr->create_trace();

		$this->_cost_fct = $cost_fct;
		$this->_routes_cache = array();

		return $r;
	}

	/*
	 * Set the heuristic function name
	 *
	 * This function must accept the following parameters :
	 *  + node_list
	 *  + start_node
	 *  + stop_node
	 *  + node
	 *	+ next_node
	 *
	 * It must return an heuristic value assigned to the selected road.
	 *
	 * The following extra field is available in the node_list :
	 *  + father : the previous node
	 *
	 * @access		public
	 * @param		array					$node_list				Node list
	 * @return		object DftkDaTrace		$r						Trace
	 */
	function &set_heuristic_fct($heuristic_fct)
	{
		$r =& $this->_tracemgr->create_trace();

		$this->_heuristic_fct = $heuristic_fct;
		$this->_routes_cache = array();

		return $r;
	}

	/*
	 * Set the successor selection function name
	 *
	 * This function must accept the following parameters :
	 *  + node_list
	 *  + successor_list
	 *	+ node
	 *
	 * It must choose one of the possible successors and return its index number.
	 * If the number returned is out of range, the first one in the list is used.
	 *
	 * If not specified the default is to take the first one in the list.
	 *
	 * @access		public
	 * @param		array					$node_list				Node list
	 * @return		object DftkDaTrace		$r						Trace
	 */
	function &set_successor_selection_fct($successor_selection_fct)
	{
		$r =& $this->_tracemgr->create_trace();

		$this->_successor_selection_fct = $successor_selection_fct;
		$this->_routes_cache = array();

		return $r;
	}

	/**
	 * Initialize private information in the node list
	 *
	 * @access		private
	 */
	function _init()
	{
		foreach ($this->_node_list as $key => $v)
		{
			$this->_node_list[$key]['father'] = null;
			$this->_node_list[$key]['f'] = 0;
			$this->_node_list[$key]['g'] = 0;
		}
	}

	/**
	 * Get the path to go from one node to another
	 *
	 * @access		public
	 * @param		string					$start_node				Start node
	 * @param		string					$stop_node				Stop node
	 * @return		object DftkDaTrace		$r						Trace
	 */
	function &get_path($start_node, $stop_node)
	{
		$r =& $this->_tracemgr->create_trace();

		if (count($this->_node_list)==0)
			$r->add_event('dftk-ai_missnodelist');
		if ($this->_cost_fct=="")
			$r->add_event('dftk-ai_misscostfct');
		if ($this->_heuristic_fct=="")
			$r->add_event('dftk-ai_missheuristicfct');

		if (!array_key_exists($start_node, $this->_node_list))
			$r->add_event('dftk-ai_badstartnode', $start_node);
		if (!array_key_exists($stop_node, $this->_node_list))
			$r->add_event('dftk-ai_badstopnode', $stop_node);

		if (!$r->has_error())
		{
			$this->_start_node = $start_node;
			$this->_stop_node = $stop_node;

			if ($this->_routes_cache[$this->_start_node][$this->_stop_node])
				$r->set_result('path', $this->_routes_cache[$this->_start_node][$this->_stop_node]);
			else if ($this->_routes_cache[$this->_stop_node][$this->_start_node])
				$r->set_result('path', array_reverse($this->_routes_cache[$this->_stop_node][$this->_start_node]));
			else
			{
				$this->_routes_cache[$this->_start_node][$this->_stop_node] = $this->_find_path($this->_node_list, $this->_start_node, $this->_stop_node);
				$r->set_result('path', $this->_routes_cache[$this->_start_node][$this->_stop_node]);
			}
		}

		return $r;
	}

	/**
	 * Compute one of the best path to go from one node to another
	 *
	 * @access		private
	 * @return		array					$r						Path
	 */
	function _find_path()
	{
		$this->_init();

		$P = array($this->_start_node);
		$Q = array();
		$node = $this->_start_node;
		$cost = $this->_cost_fct;
		$heuristic = $this->_heuristic_fct;

		while ((count($P)!=0) && ($node!=$this->_stop_node))
		{
			$v = $this->_select_successor(array_merge($P, $Q), $node);
			if (!$v)
			{
				//print("finished with node : ".$node."\n");
				$P = dftk_array_del_value($P, $node);
				//print("P : ");
				//print_r($P);
				array_push($Q, $node);
				//print("Q : ");
				//print_r($Q);
			}
			else
			{
				$next_node = $v[0];
				//print("current node : ".$node."\n");
				//print("next node : ".$next_node."\n");
				$this->_node_list[$next_node][g] = $this->_node_list[$node][g]+$cost($this->_node_list, $node, $next_node);
				$this->_node_list[$next_node][f] = $this->_node_list[$next_node][g] + $heuristic($this->_node_list, $this->_start_node, $this->_stop_node, $node, $next_node);
				$this->_node_list[$next_node][father] = array($node, $v[1]);
				$P = $this->_array_sorted_insert($P, $next_node);
				//print("P : ");
				//print_r($P);
			}
			if (count($P)!=0)
				$node = $P[0];
		}

		if (count($P)==0)
			return null;
		return $this->_path($node);
	}

	/**
	 * Search the possible successors for the current node, and choose one
	 *
	 * @access		private
	 * @param		array					$union					Nodes not to explore
	 * @param		string					$node					Current node name
	 * @return		string					$r						Name of the choosen node
	 */
	function _select_successor($union, $node)
	{
		$S = array();
		$cost = $this->_cost_fct;

		if (count($this->_node_list[$node][successors])>0)
			foreach ($this->_node_list[$node][successors] as $v)
			{
				//print ("possible successor : ".$v[0]."   g(v)=".$this->_node_list[$v[0]][g]."   g(u)=".$this->_node_list[$node][g]."   cost=".$cost($this->_node_list, $node, $v[0])."\n");
				if ((!in_array($v[0], $union)) || ($this->_node_list[$v[0]][g] > $this->_node_list[$node][g]+$cost($this->_node_list, $node, $v[0])))
					$S[] = $v;
			}
	
		if (count($S) == 0)
			$r = null;
		elseif ($this->_successor_selection_fct!="")
		{
			$successor_selection_fct = $this->_successor_selection_fct;
			$i = $successor_selection_fct($this->_node_list, $S, $node);
			if (($i<0) || ($i>=count($S)))
				$i = 0;
			$r = $S[$i];
		}
		else
			$r = $S[0];

		// choix du successeur
		//print("successeur : ");
		//print_r($r);
		return $r;
	}

	/**
	 * Insert a value in a sorted array
	 *
	 * This function ensure the array will stay sorted after each insert.
	 *
	 * @internal Cannot use dftk_array_sorted_insert as the sort function is in fact a non-static method
	 *
	 * @access		private
	 * @param		array				$tab_orig				Array to process
	 * @param		mixed				$value					Value to insert
	 * @return		array				$tab					Array processed
	 */
	function _array_sorted_insert($tab_orig, $value)
	{
		$n = count($tab_orig);
	
		for ($i=0; $i<count($tab_orig); $i++)
			if ($this->_test_func($value, $tab_orig[$i]))
			{
				$n = $i;
				break;
			}
	
		return dftk_array_insert($tab_orig, $n, $value);
	}

	/**
	 * Sort function for P
	 *
	 * @access		private
	 * @param		string					$state1					Node 1
	 * @param		string					$state2					Node 2
	 * @return		boolean					$r						true if node 1 comes before node 2
	 */
	function _test_func($state1, $state2)
	{
		$f1 = $this->_node_list[$state1][f];
		$f2 = $this->_node_list[$state2][f];

		if ($f1 < $f2)
			return true;
		if ($f1 > $f2)
			return false;

		$g1 = $this->_node_list[$state1][g];
		$g2 = $this->_node_list[$state2][g];

		if ($g1 > $g2)
			return true;
		if ($g1 < $g2)
			return false;

		if ($state1 == $this->_stop_node)
			return true;

		return false;
	}

	/**
	 * Retrieve the path found to the specified node
	 *
	 * The path is an array of couples (<node_name>, <road_name>) to follow
	 * from the start_node to reach the stop_node..
	 *
	 * @access		private
	 * @param		string					$node					Node
	 * @return		array					$r						Path
	 */
	function _path($node)
	{
		$father = $this->_node_list[$node][father];
		if ($father)
			return array_merge($this->_path($father[0]), array(array($node, $father[1])));
		else
			return array();
	}

}

?>
